/*
 * Decompiled with CFR 0.152.
 */
package com.paterva.maltego.util;

import com.paterva.maltego.util.ImageUtils;
import com.paterva.maltego.util.StringUtilities;
import com.paterva.maltego.util.Version;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.imageio.ImageIO;
import org.apache.commons.io.FilenameUtils;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.modules.Places;
import org.openide.util.Exceptions;

public class FileUtilities {
    private static final Logger LOG = Logger.getLogger(FileUtilities.class.getName());
    public static final String TYPE_IMAGE = "Image";
    public static final String TYPE_TEXT = "Text";
    public static final String TYPE_BINARY = "Binary";
    private static final String ENV_ALLUSERSPROFILE = "ALLUSERSPROFILE";

    private FileUtilities() {
    }

    public static void writeTo(InputStream is, File file) throws IOException {
        Files.copy(is, file.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    public static void copy(File file, OutputStream os) throws IOException {
        try (FileInputStream is = new FileInputStream(file);){
            FileUtil.copy((InputStream)is, (OutputStream)os);
        }
    }

    public static FileObject getOrCreate(String path) throws IOException {
        return FileUtilities.getOrCreate(FileUtil.getConfigRoot(), path);
    }

    public static FileObject getOrCreate(FileObject folder, String path) throws IOException {
        String[] components;
        for (String f : components = path.split("/")) {
            folder = FileUtilities.getOrCreateSingle(folder, f);
        }
        return folder;
    }

    private static FileObject getOrCreateSingle(FileObject base, String folder) throws IOException {
        FileObject fo = base.getFileObject(folder);
        if (fo == null && base.canWrite()) {
            fo = base.createFolder(folder);
        }
        return base.getFileObject(folder);
    }

    public static String getFileExtension(File file) {
        if (file == null) {
            return null;
        }
        return FileUtilities.getFileExtension(file.getName());
    }

    public static String getFileExtension(String filename) {
        if (filename == null) {
            return null;
        }
        int pos = filename.lastIndexOf(46);
        if (pos < 0 || pos >= filename.length() - 1) {
            return "";
        }
        return filename.substring(pos + 1);
    }

    public static String getFileExtensionCommonsIO(String filename) {
        return FilenameUtils.getExtension((String)filename);
    }

    public static String getFileNameWithoutExtension(String fileNameWithExt) {
        return FilenameUtils.removeExtension((String)fileNameWithExt);
    }

    public static File getUserHome() {
        String home = System.getProperty("user.home");
        return new File(home);
    }

    public static File getUserDir(boolean allUsers) {
        if (allUsers) {
            File settings = new File(FileUtilities.getEnvVar(ENV_ALLUSERSPROFILE), "Paterva");
            settings = new File(settings, Version.getCurrent().getAppnameCapital());
            settings.mkdirs();
            settings.setWritable(true, false);
            return settings;
        }
        File userDir = Places.getUserDirectory();
        return userDir;
    }

    public static void windowsFullControl(File file) {
    }

    private static String getEnvVar(String folderName) {
        String path = System.getenv(folderName);
        if (folderName.equalsIgnoreCase(ENV_ALLUSERSPROFILE) && FileUtilities.isWindowsXP()) {
            path = path.concat(System.getProperty("file.separator"));
            path = path.concat(FileUtilities.getAppdataFolderName());
        }
        return path;
    }

    public static boolean isWindows() {
        String os = System.getProperty("os.name");
        return os.toLowerCase().contains("windows");
    }

    private static boolean isWindowsXP() {
        String os = System.getProperty("os.name");
        return os.contains("Windows XP");
    }

    private static String getAppdataFolderName() {
        String userAppdata = System.getenv("APPDATA");
        String[] parts = userAppdata.split("\\" + System.getProperty("file.separator"));
        return parts[parts.length - 1];
    }

    public static FileObject createUniqueFile(FileObject folder, String filename, String extension) throws IOException {
        String newName;
        FileObject file;
        filename = filename.toLowerCase();
        int index = 0;
        do {
            newName = index > 0 ? filename + index : filename;
            newName = newName + "." + extension;
            file = folder.getFileObject(newName);
            ++index;
        } while (file != null);
        return folder.createData(newName);
    }

    public static FileObject createFreeFileObjectData(FileObject parentFileObject, String typeName, String ext) throws IOException {
        FileObject freeFileObject = null;
        typeName = FileUtilities.createFreeFileObjectName(parentFileObject, typeName, ext);
        freeFileObject = parentFileObject.createData(typeName, ext);
        return freeFileObject;
    }

    public static String createFreeFileObjectName(FileObject parentFileObject, String typeName, String ext) {
        File fileCaseInsensitive = FileUtilities.getFileFromFileObject(parentFileObject, typeName, ext);
        if (fileCaseInsensitive.exists()) {
            typeName = FileUtil.findFreeFileName((FileObject)parentFileObject, (String)typeName, (String)ext);
        }
        return typeName;
    }

    public static File getFileFromFileObject(FileObject parentFileObject, String typeName, String ext) {
        File fileCaseInsensitive = new File(FileUtil.toFile((FileObject)parentFileObject), typeName + "." + ext);
        return fileCaseInsensitive;
    }

    public static String replaceIllegalChars(String filename) {
        return filename.replaceAll("[^A-Za-z0-9_ ]", "_");
    }

    public static String replaceIllegalCharsKeepDots(String filename) {
        return filename.replaceAll("[^A-Za-z0-9_ .]", "_");
    }

    public static String replaceIllegalCharsForPandora(String filename) {
        return filename.replaceAll("[.|\\[\\]*]", "_");
    }

    public static String removePathChangeCharacters(String fileOrFolderName) {
        if (StringUtilities.isNullOrEmpty(fileOrFolderName)) {
            return fileOrFolderName;
        }
        return fileOrFolderName.replace("~", "").replace("/", "").replace("\\", "").replaceAll("[.][.]+", "");
    }

    public static String createUniqueLegalFilename(Collection<String> otherNames, String preferedName) {
        preferedName = preferedName.toLowerCase();
        return StringUtilities.createUniqueString(otherNames, FileUtilities.replaceIllegalChars(preferedName));
    }

    public static void copyFile(File sourceFile, File destFile) throws IOException {
        long count = 0xA00000L;
        if (!destFile.exists()) {
            destFile.createNewFile();
        }
        try (FileInputStream fIn = new FileInputStream(sourceFile);
             FileChannel source = fIn.getChannel();
             FileOutputStream fOut = new FileOutputStream(destFile);
             FileChannel destination = fOut.getChannel();){
            long bytes = source.size();
            for (long transfered = 0L; transfered < bytes; transfered += destination.transferFrom(source, transfered, 0xA00000L)) {
            }
        }
    }

    public static File getTempDir(String subDir) {
        return new File(FileUtilities.getMaltegoTempDir(), subDir);
    }

    public static File createTempDir(String subDir) {
        File cacheDir = FileUtilities.getTempDir(subDir);
        try {
            cacheDir.mkdirs();
        }
        catch (Exception ex) {
            Exceptions.printStackTrace((Throwable)ex);
        }
        return cacheDir;
    }

    public static File getMaltegoTempDir() {
        String tempDir = System.getProperty("java.io.tmpdir");
        Version version = Version.getCurrent();
        String appnameCapital = version.getAppnameCapital();
        String suffix = version.toString(false, true, false);
        return new File(tempDir, appnameCapital + suffix);
    }

    public static String getFileType(String filename) {
        Object[] textTypes = new String[]{"csv", "htm", "html", "ini", "text", "txt"};
        Object[] imgTypes = ImageIO.getReaderFileSuffixes();
        Arrays.sort(textTypes);
        Arrays.sort(imgTypes);
        String ext = FileUtilities.getFileExtension(filename).toLowerCase();
        if (Arrays.binarySearch(textTypes, ext) >= 0) {
            return TYPE_TEXT;
        }
        if (Arrays.binarySearch(imgTypes, ext) >= 0) {
            return TYPE_IMAGE;
        }
        return TYPE_BINARY;
    }

    public static boolean isPossibleBase64Data(String data) {
        int length;
        boolean possibleBase64 = false;
        if (data != null && (length = data.trim().length()) >= 100 && length != 128 && length != 256) {
            try {
                data = ImageUtils.removeBase64Prefix(data).trim();
                Pattern pattern = Pattern.compile("^([A-Za-z0-9+/]{4})*([A-Za-z0-9+/]{4}|[A-Za-z0-9+/]{3}=|[A-Za-z0-9+/]{2}==)$");
                Matcher matcher = pattern.matcher(data);
                possibleBase64 = matcher.find();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return possibleBase64;
    }

    public static boolean delete(File f) {
        if (!f.exists()) {
            return true;
        }
        FileUtilities.deleteContents(f);
        return f.delete();
    }

    public static void deleteContents(File folder) {
        if (folder.isDirectory()) {
            for (File c : folder.listFiles()) {
                if (FileUtilities.delete(c)) continue;
                LOG.log(Level.SEVERE, "Failed to delete {0}", c);
            }
        }
    }

    public static List<File> removeDirectories(List<File> files) {
        if (files == null) {
            return null;
        }
        ArrayList<File> filesOnly = new ArrayList<File>();
        for (File file : files) {
            if (file.isDirectory()) continue;
            filesOnly.add(file);
        }
        return filesOnly;
    }

    public static File createRandomDir(File parent) {
        Random r = new Random();
        File dir = null;
        while (dir == null || dir.exists()) {
            dir = new File(parent, Integer.toString(r.nextInt(1000000)));
        }
        dir.mkdirs();
        return dir;
    }

    public static String insertBeforeSuffix(String path, String text) {
        String withoutSuffix = path;
        String suffix = "";
        if (path.lastIndexOf(46) >= 0) {
            withoutSuffix = path.substring(0, path.lastIndexOf(46));
            suffix = path.substring(path.lastIndexOf(46), path.length());
        }
        return withoutSuffix + text + suffix;
    }

    public static boolean isRemoteFileURL(URL url) {
        if (url == null) {
            return false;
        }
        String protocol = url.getProtocol();
        String host = url.getHost();
        return "file".equalsIgnoreCase(protocol) && !StringUtilities.isNullOrEmpty(host) && !"localhost".equals(host) && !"127.0.0.1".equals(host);
    }
}

